unit FindMoviesImpl;

{
  Invokable implementation File for TFindMovies which implements IFindMovies.

  Copyright  Keith Wood (kbwood@iprimus.com.au)
  Written 17 September, 2002.
}

interface

uses
  InvokeRegistry, Types, XSBuiltIns, FindMoviesIntf, DB, DBTables;

type

  { TFindMovies }
  TFindMovies = class(TInvokableClass, IFindMovies)
  private
    FDataSource: TDataSource;
    FDetailQuery: TQuery;
    FRatingQuery: TQuery;
    FStarsQuery: TQuery;
  public
    constructor Create; override;
    destructor Destroy; override;
    function FindMovies(const Rating: TMovieRating): TStringDynArray; stdcall;
    function GetMovieDetails(const MovieName: string): TMovieDetails; stdcall;
  end;

implementation

const
  DBName = 'Movie-watcher';  // BDE alias

{ TFindMovies -----------------------------------------------------------------}

{ Initialisation of parameterised queries }
constructor TFindMovies.Create;
begin
  inherited Create;
  FDetailQuery := TQuery.Create(nil);
  with FDetailQuery do
  begin
    DatabaseName := DBName;
    SQL.Add('select movie_id, name, length_mins, rating, director, synopsis, url');
    SQL.Add('from movie');
    SQL.Add('where name = :name');
    ParamByName('name').DataType := ftString;
    Prepare;
  end;
  FDataSource         := TDataSource.Create(nil);
  FDataSource.DataSet := FDetailQuery;
  FRatingQuery := TQuery.Create(nil);
  with FRatingQuery do
  begin
    DatabaseName := DBName;
    SQL.Add('select name');
    SQL.Add('from movie');
    SQL.Add('where :rating = '''' or :rating = rating');
    ParamByName('rating').DataType := ftString;
    Prepare;
  end;
  FStarsQuery := TQuery.Create(nil);
  with FStarsQuery do
  begin
    DatabaseName := DBName;
    SQL.Add('select star');
    SQL.Add('from stars');
    SQL.Add('where movie_id = :movie_id');
    DataSource := FDataSource;  // Link to detail query
    Prepare;
  end;
end;

{ Release resources }
destructor TFindMovies.Destroy;
begin
  FDataSource.Free;
  FDetailQuery.Free;
  FRatingQuery.Free;
  FStarsQuery.Free;
  inherited Destroy;
end;

{ Find movies with a given rating, or all movies }
function TFindMovies.FindMovies(const Rating: TMovieRating): TStringDynArray;
var
  Index: Integer;
begin
  with FRatingQuery do
    try
      ParamByName('rating').AsString := Ratings[Rating];
      Open;
      if RecordCount = 0 then
        raise EFindMoviesException.Create('for rating ' + Ratings[Rating]);

      SetLength(Result, RecordCount);
      Index := 0;
      while not EOF do
      begin
        Result[Index] := FieldByName('name').AsString;
        Inc(Index);
        Next;
      end;
    finally
      Close;
    end;
end;

{ Retrieve the details for a single movie identified by name }
function TFindMovies.GetMovieDetails(const MovieName: string): TMovieDetails;
var
  Index: Integer;
  Rating: TMovieRating;
  Stars: TStringDynArray;
begin
  with FDetailQuery do
    try
      ParamByName('name').AsString := MovieName;
      Open;
      if RecordCount = 0 then
        raise EFindMoviesException.Create('named ' + MovieName);

      Result          := TMovieDetails.Create;
      Result.Name     := FieldByName('name').AsString;
      Result.Length   := FieldByName('length_mins').AsInteger;
      Result.Director := FieldByName('director').AsString;
      Result.Synopsis := FieldByName('synopsis').AsString;
      Result.URL      := FieldByName('url').AsString;
      for Rating := High(TMovieRating) downto Succ(Low(TMovieRating)) do
        if FieldByName('rating').AsString = Ratings[Rating] then
          Break;
      Result.Rating   := Rating;

      // Load child records
      with FStarsQuery do
        try
          Open;
          SetLength(Stars, RecordCount);
          Index := 0;
          while not EOF do
          begin
            Stars[Index] := FieldByName('star').AsString;
            Inc(Index);
            Next;
          end;
        finally
          Close;
        end;
      Result.Stars := Stars;
    finally
      Close;
    end;
end;

initialization
  { Invokable classes must be registered }
  InvRegistry.RegisterInvokableClass(TFindMovies);
end.
